import { Router } from 'express';
import { body } from 'express-validator';
import { vendorController } from './controller';
import { authenticateUser, requireVendor, requireAdmin } from '@/middlewares/auth';
import { validateRequest } from '@/middlewares/validation';

const router = Router();

// Validation rules
const onboardingValidation = [
  body('business_name')
    .isLength({ min: 2, max: 100 })
    .withMessage('Business name must be between 2 and 100 characters'),
  body('profile_picture_url')
    .optional()
    .isURL()
    .withMessage('Profile picture must be a valid URL')
];

const profileUpdateValidation = [
  body('business_name')
    .optional()
    .isLength({ min: 2, max: 100 })
    .withMessage('Business name must be between 2 and 100 characters'),
  body('profile_picture_url')
    .optional()
    .isURL()
    .withMessage('Profile picture must be a valid URL'),
  body('subscription_plan')
    .optional()
    .isIn(['free', 'basic', 'premium', 'enterprise'])
    .withMessage('Invalid subscription plan')
];

/**
 * @route   GET /api/vendors/me
 * @desc    Get current vendor profile
 * @access  Private (Vendor)
 */
router.get('/me', authenticateUser, requireVendor, vendorController.getMyProfile);

/**
 * @route   POST /api/vendors/onboarding
 * @desc    Complete vendor onboarding
 * @access  Private (Vendor)
 */
router.post('/onboarding', 
  authenticateUser, 
  requireVendor, 
  onboardingValidation, 
  validateRequest, 
  vendorController.completeOnboarding
);

/**
 * @route   PUT /api/vendors/profile
 * @desc    Update vendor profile
 * @access  Private (Vendor)
 */
router.put('/profile', 
  authenticateUser, 
  requireVendor, 
  profileUpdateValidation, 
  validateRequest, 
  vendorController.updateProfile
);

/**
 * @route   GET /api/vendors/:id
 * @desc    Get vendor by ID
 * @access  Private (Admin)
 */
router.get('/:id', authenticateUser, requireAdmin, vendorController.getVendorById);

/**
 * @route   GET /api/vendors
 * @desc    Get all vendors
 * @access  Private (Admin)
 */
router.get('/', authenticateUser, requireAdmin, vendorController.getAllVendors);

export default router; 